/********************************************************************************
* @file		:jq8400.c
* @brief	:jq8400 语音播报模块驱动
* @Author	:Gwen9
* @Date		:2024/08/12
* @Version	:V1.0 初始版本
*******************************************************************************/
#include "driver_conf.h"

#ifdef DRIVER_JQ8400_ENABLED

/* Private Macros ----------------------------------------------------------*/
typedef struct _S_JQ8400
{
	my_uart_id_t uart_id;
}s_jq8400;

/* Private Typedef  -------------------------------------------------------------------*/
/*基本指令集，直发送不接收，两线串口模式*/
const uint8_t jq8400_cmd_list[][6] = 
{
	/*播放选择*/
	{0xAA , 0x0B , 0x01 , 0x00 , 0xB6},   	/*播放U盘*/
	{0xAA , 0x0B , 0x01 , 0x01 , 0xB7},   	/*播放SD卡*/
	{0xAA , 0x0B , 0x01 , 0x02 , 0xB8},   	/*播放FLASH*/

	/*播放控制*/
	{0xAA, 0x02, 0x00, 0xAC},          		/*播放:在任何时候发此命令都会从头开始播放当前曲目*/
	{0xAA, 0x03, 0x00, 0xAD},          		/*暂停播放*/
	{0xAA, 0x04, 0x00, 0xAE},          		/*停止播放*/
	{0xAA, 0x05, 0x00, 0xAF},          		/*上一曲*/
	{0xAA, 0x06, 0x00, 0xB0},          		/*下一曲*/
	{0xAA, 0x07, 0x02, 0x00, 0x01, 0xB4}, 	/*指定曲目播放*/
	{0xAA, 0x0E, 0x00, 0xB8},          		/*上一目录*/
	{0xAA, 0x0F, 0x00, 0xB9},          		/*下一目录*/
	{0xAA, 0x10, 0x00, 0xBA},          		/*结束播放：此指令可以提前结束当前操作，会结束当前播放，如果是在插播等则会提前结束插播返回原来状态*/
	{0xAA, 0x22, 0x02, 0x00, 0x05, 0xD3},	/*快退 5秒*/
	{0xAA, 0x23, 0x02, 0x00, 0x05, 0xD4}, 	/*快进 5秒*/

	/*播放模式*/
	{0xAA, 0x18, 0x01, 0x00, 0xC3},			/*全部循环*/
	{0xAA, 0x18, 0x01, 0x01, 0xC4},   		/*单曲循环*/
	{0xAA, 0x18, 0x01, 0x02, 0xC5},   		/*单曲停止*/
	{0xAA, 0x18, 0x01, 0x03, 0xC6},   		/*随机播放*/
	{0xAA, 0x18, 0x01, 0x04, 0xC7},   		/*目录循环*/
	{0xAA, 0x18, 0x01, 0x05, 0xC8},   		/*目录随机*/
	{0xAA, 0x18, 0x01, 0x06, 0xC9},   		/*目录停止*/
	{0xAA, 0x18, 0x01, 0x07, 0xCA},   		/*顺序播放*/

	/*播放音量*/
	{0xAA, 0x14, 0x00, 0xBE},				/*音量加*/
	{0xAA, 0x15, 0x00, 0xBF},				/*音量减*/
	{0xAA, 0x13, 0x01, 0x14, 0xD2},			/*音量设置 默认20*/

	/*系统信息: 需要使能接收, 否则只需要一个tx引脚即可*/
	{0xAA, 0x01, 0x00, 0xAB},       		/*获取状态*/
};

/* Private Functions Declare ----------------------------------------------------------*/
static int jq8400_send_cmd(const c_jq8400 *p_obj, jq8400_cmd_t cmd, uint16_t param);

/* Private Variables ------------------------------------------------------------------*/
/* Public Functions -------------------------------------------------------------------*/
c_jq8400 jq8400_create(my_uart_id_t uart_id)
{
	c_jq8400 new = {0};	//新建一个空的GPIO对象		
	int ret;

	/*参数检测*/
	if(uart_id >= UART_MAX){
		return new;
	}
	/*为 JQ8400 新对象的私有成员申请内存 同时清空内存*/
	new.this = MY_MALLOC(sizeof(s_jq8400));
	if(NULL == new.this){
		return new;
	}
	memset(new.this, 0, sizeof(s_jq8400));

	/*初始化 JQ8400 对象的私有成员*/
	((s_jq8400*)new.this)->uart_id = uart_id;

	/*初始化 JQ8400 对象的公有接口*/
	new.send 				= jq8400_send_cmd;

	/*初始化 JQ8400 用到的串口*/
	ret = my_uart.init(uart_id, JQ8400_BAUDRATE, UART_MODE_ISR);
	if(ret != R_OK)		goto errorHandle;

	/*设置JQ8400 默认工作参数*/
	ret = jq8400_send_cmd(&new, JQ8400_DEFAULT_DISK, 0);			//盘符
	if(ret != R_OK)		goto errorHandle;
	
	ret =  jq8400_send_cmd(&new, JQ8400_DEFAULT_MODE, 0);			//播放模式
	if(ret != R_OK)		goto errorHandle;
	
	ret =  jq8400_send_cmd(&new, JQ_VOL_SET, JQ8400_DEFAULT_VOL);	//默认音量
	if(ret != R_OK)		goto errorHandle;
	
	return new;

errorHandle:
	new.this = NULL;
	return new;
}

/* Private Functions -------------------------------------------------------------------*/
static int jq8400_send_cmd(const c_jq8400 *p_obj, jq8400_cmd_t cmd, uint16_t param)
{
    s_jq8400* this = NULL;
	uint8_t send_len;
	uint8_t send_buf[6];
	int ret;
    /*检查参数*/
    if(NULL == p_obj || NULL == p_obj->this) {
        return R_NULL;
    }
	if(cmd >= JQ_ERROR){
		return R_ERROR;
	}
    this = p_obj->this;

	/*装载命令到发送缓存*/
    memcpy(send_buf, &jq8400_cmd_list[cmd][0], 6);
	if(cmd == JQ_VOL_SET){																	//5个字节的带参数命令
		send_len = 5;
		send_buf[3] = param;
		send_buf[4] = send_buf[0] + send_buf[1] + send_buf[2] + send_buf[3];
	}else if(cmd == JQ_CTRL_PLAY_SPECIFY || cmd == JQ_CTRL_REWIND || cmd == JQ_CTRL_FORWARD){	//6个字节的带参数命令
		send_len = 6;
		send_buf[3] = param >> 8;
		send_buf[4] = param & 0x00FF;
		send_buf[5] = send_buf[0] + send_buf[1] + send_buf[2] + send_buf[3] + send_buf[4];
	}else{																					//其他不带参数的命令
		if(cmd <= JQ_DISK_FLASH)		send_len = 5;	//播放盘符选择命令
		else if(cmd <= JQ_CTRL_FORWARD)	send_len = 4;	//播放控制命令
		else if(cmd <= JQ_MODE_ORDER)	send_len = 5;	//播放模式选择命令
		else if(cmd <= JQ_VOL_SET)		send_len = 4;	//播放音量控制命令
		else if(cmd < JQ_ERROR)			send_len = 4;	//获取状态信息命令
		else							return R_ERROR;
	}
	/*发送命令*/
	ret = my_uart.send(this->uart_id, send_buf, send_len);
	if(ret != R_OK)		return R_ERROR;

	return R_OK;
}

#endif

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/


